/// @file udev.cpp
///
/// @brief Implements the @ref Udev class
///
/// @description The udev class aims to interface with the udev library,
/// gathering information from devices and monitoring events.
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see EventHandler DeviceDetector DeviceInfo

#include <algorithm>
#include <iostream>
#include <list>
#include <libudev.h>
#include <cstring>

#include <sys/epoll.h>

#include "eventHandler.h"
#include "deviceDetector.h"
#include "logger.h"
#include "udev.h"

namespace adit {
namespace uspi {

extern std::list<DeviceInfo> lDevInfo;
extern const char *dev_sysattr_str[DSYS_NONE + 1];

/* *************  strings definition  ************* */
/// @defgroup UDEV_STR Strings for udev handling
/// @{
#define STR_UDEV                "udev"          ///< udev
#define STR_USB                 "usb"           ///< usb subsystem
#define STR_USB_DEVICE          "usb_device"    ///< usb_device device type
#define STR_BLUETOOTH           "bluetooth"     ///< bluetooth subsystem
#define STR_ADD                 "add"           ///< Add event
#define STR_REMOVE              "remove"        ///< Remove event
#define STR_CHANGE              "change"        ///< Change event
/// @}

/// @brief Constructor
///
/// Will initialize @ref mUdev with a new udev structure, @ref mMon with a
/// monitor structure, add the subsystem to match and enable the monitor. If any
/// of these step fails, throw an error.
/// Once @ref mFd is initialized with the monitor file descriptor, look for all
/// currently available devices.
Udev::Udev():
    mUdev(nullptr),
    mMon(nullptr),
    mFd(-1),
    mDevDetector(nullptr)
{
    mUdev = udev_new();

    if (!mUdev)
        throw std::bad_alloc();

    mMon = udev_monitor_new_from_netlink(mUdev, STR_UDEV);

    if (!mMon)
        throw std::bad_alloc();

    if (udev_monitor_filter_add_match_subsystem_devtype(mMon,
                                                        STR_USB,
                                                        STR_USB_DEVICE))
        throw std::runtime_error("Can't attach USB filter to monitor");

    if (udev_monitor_filter_add_match_subsystem_devtype(mMon,
                                                        STR_BLUETOOTH,
                                                        NULL))
        throw std::runtime_error("Can't attach BT filter to monitor");

    if (udev_monitor_enable_receiving(mMon))
        throw std::runtime_error("Can't enable receiving.");

    mFd = udev_monitor_get_fd(mMon);

}

/// @brief Destructor
///
/// Releases the monitor and the udev resources.
Udev::~Udev()
{
    if (mMon) {
        ddlog << "Destroying monitor" << std::endl;
        udev_monitor_unref(mMon);
    }

    if (mUdev) {
        ddlog << "Destroying udev" << std::endl;
        udev_unref(mUdev);
    }
}

/// @brief Handle a udev device event.
/// @param dev The device that generated the event
/// @param action The action issued ("add", "remove", "change")
/// @param check If the device must be checked for DevDetector notification
void Udev::handleDevice(struct udev_device *dev,
                        std::string &action,
                        bool check)
{
    DeviceInfo dInfo;
    const char *node = udev_device_get_devnode(dev);
    const char *dev_path = udev_device_get_devpath(dev);
    enum DD_EVENT_TYPE eType = DD_EV_ADD;

    if (!node && !dev_path) {
        ddlog << "No node nor dev path found." << std::endl;
        return;
    }

    if (node)
        dInfo.setNode(node);

    if (dev_path)
        dInfo.setDevPath(dev_path);

    if (action == STR_REMOVE) {
        eType = DD_EV_REMOVE;
        std::list<DeviceInfo>::iterator it = std::find(lDevInfo.begin(),
                                                       lDevInfo.end(),
                                                       dInfo);
        if (it != std::end(lDevInfo)) {
            dInfo = *it;

            lDevInfo.erase(it);
        }
        ddlog << dInfo.getDevPath() << " removed." << std::endl;

        goto exit;
    } else if (action == STR_CHANGE) {
        eType = DD_EV_CHANGE;
    }

    for (int i = DSYS_STRINGS; i < DSYS_NONE; ++i) {
        const char *tmp = udev_device_get_sysattr_value(dev,
                                                        dev_sysattr_str[i]);
        dInfo.setDeviceInfo(static_cast<enum dev_sysattr>(i), tmp);
    }

    // Ignore system devices
    if ((dInfo.getiDeviceInfo(DSYS_DEVPATH) == 0) &&
        (dInfo.getDeviceInfo(DSYS_ADDRESS) == ""))
        return;

    lDevInfo.push_back(dInfo);

    ddlog << dInfo.getDevPath() << " added." << std::endl;

    if (check)
        dInfo = dInfo.check();

exit:
    if (check)
        mDevDetector->checkDevice(dInfo, eType);
}

/// @brief Callback to handle udev events
///
/// This function is executed by the EventHandler, or through the
/// Udev::dispatchEvent method.
static int handle_event()
{
    Udev& u = Udev::getInstance();
    std::string action;
    struct udev_device *dev = u.monitorReceiveDevice();

    if (!dev) {
        ddlog << "No Device from receive_device(). An error occurred."
            << std::endl;
        return 0;
    }

    action.assign(udev_device_get_action(dev));

    u.handleDevice(dev, action);

    udev_device_unref(dev);

    return 0;
}

/// Dispatch events received on the @ref mFd file descriptor.
int Udev::dispatchEvent()
{
    return handle_event();
}

/// @brief Check each parent of a specific subsystem ("usb" or "bluetooth").
/// @param enumerate The udev_enumerate structure to use for enumeration.
/// @param ssystem The subsystem to be filtered.
///
/// For each entry of the enumeration, this method look for a valid udev device.
/// The corresponding device is added to the device list. It then looks for
/// parent of this device, and add them if they are valid.
void Udev::checkParent(struct udev_enumerate *enumerate, const char *ssystem)
{
    struct udev_list_entry *devices, *dev_list_entry;
    struct udev_device *dev;
    struct udev_device *org;
    std::string action = STR_ADD;

    udev_enumerate_add_match_subsystem(enumerate, ssystem);
    udev_enumerate_scan_devices(enumerate);
    devices = udev_enumerate_get_list_entry(enumerate);
    udev_list_entry_foreach(dev_list_entry, devices) {
        const char *path;
        struct udev_device *tmp;

        path = udev_list_entry_get_name(dev_list_entry);
        org = dev = udev_device_new_from_syspath(mUdev, path);

        do {
            if (dev)
                handleDevice(dev, action, false);
            else
                break;

            tmp = udev_device_get_parent_with_subsystem_devtype(dev,
                                                                ssystem,
                                                                NULL);
            if (tmp)
                dev = tmp;
        } while (tmp);

        if (!dev) {
            ddlog << "Unable to find parent usb device." << std::endl;
            break;
        }

        if (org)
            udev_device_unref(org);
    }

    udev_enumerate_add_match_subsystem(enumerate, "");
}

/// @brief Put the object in standalone mode
///
/// The object registers itself to the @ref EventHandler singleton.
void Udev::setStandAlone()
{
    void *cb = (void *)&handle_event;
    EventHandler& eh = EventHandler::getInstance();

    eh.registerEvent(mFd, cb);
}

/// @brief Receive device from the udev monitor.
///
/// Only returns if there is a device available.
struct udev_device *Udev::monitorReceiveDevice(void)
{
    return udev_monitor_receive_device(mMon);
}

/// @brief Check the complete list of devices for notification.
void Udev::checkAll(void)
{
    std::list<DeviceInfo>::iterator it;
    lDevInfo.sort();
    lDevInfo.unique();

    for (it = lDevInfo.begin(); it != lDevInfo.end(); ++it) {
        mDevDetector->checkDevice((*it).check(), DD_EV_ADD);
    }
}

} // namespace uspi
} // namespace adit
